<?php

namespace Modules\Order\Http\Requests\Api\V1;

use Arr;
use Illuminate\Validation\Rule;
use Modules\Cart\CartItem;
use Modules\Cart\Facades\PosCart;
use Modules\Core\Http\Requests\Request;
use Modules\Menu\Models\Menu;
use Modules\Order\Enums\OrderType;
use Modules\Payment\Enums\PaymentType;
use Modules\SeatingPlan\Enums\TableStatus;
use Modules\User\Enums\DefaultRole;

class CreateOrderRequest extends Request
{
    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $user = auth()->user();

        $branchId = $user->effective_branch->id;
        $menu = Menu::getActiveMenu($branchId);

        abort_if(is_null($menu), 400, __("pos::messages.menu_is_not_active"));

        $orderTypes = $menu->branch->order_types ?: [];
        $paymentMethods = $menu->branch->payment_methods;

        if ($user->hasRole(DefaultRole::Waiter->value) && in_array(OrderType::DineIn->value, $orderTypes)) {
            $orderTypes = [OrderType::DineIn->value];
        }

        return [
            "type" => ["required", Rule::in($orderTypes)],
            "table_id" => [
                "bail",
                "required_if:type,dine_in",
                "nullable",
                "numeric",
                Rule::exists("tables", "id")
                    ->whereNull("deleted_at")
                    ->where('is_active', true)
                    ->where('branch_id', $menu->branch_id)
                    ->where("status", TableStatus::Available->value),
            ],
            "payment_methods" => "nullable|array",
            "payment_methods.*" => ["required", Rule::in($paymentMethods)],
            "payments" => [
                Rule::requiredIf(fn() => count($this->input('payment_methods', [])) > 1),
                "array",
                Rule::when(
                    fn() => count($this->input('payment_methods', [])) > 1,
                    "min:2"
                ),
            ],
            "payments.*.method" => [
                "required",
                Rule::in($this->input('payment_methods', [])),
                function ($attribute, $value, $fail) {
                    $methods = Arr::pluck($this->input('payments', []), 'method');
                    if (count($methods) !== count(array_unique($methods))) {
                        $fail(__("pos::messages.invalid_payment_method"));
                    }
                }
            ],
            "payments.*.amount" => "required|numeric|min:1|max:99999999999999",
            "payment_type" => ["required", Rule::enum(PaymentType::class)],
            "amount_to_be_paid" => "required|numeric|min:0.0001",
            "pos_register_id" => "bail|required|numeric|exists:pos_registers,id,deleted_at,NULL,is_active,1,branch_id,$menu->branch_id",
            "products" => "required|array",
            "products.*" => "bail|required|numeric|exists:products,id,deleted_at,NULL,is_active,1,menu_id,$menu->id",
            "notes" => "nullable|string|max:1000",
            "guest_count" => "nullable|numeric|min:1",
            "car_plate" => "nullable|string|max:200",
            "car_description" => "nullable|string|max:200",
            "scheduled_at" => "required_if:type,pre_order|nullable|date|after_or_equal:today|date_format:Y-m-d H:i:s"];
    }

    /** @inheritDoc */
    public function validationData(): array
    {
        return [
            ...parent::validationData(),
            'type' => PosCart::orderType()->value(),
            "products" => PosCart::items()->map(fn(CartItem $cartItem) => $cartItem->product->id)->toArray(),
        ];
    }

    /** @inheritDoc */
    protected function availableAttributes(): string
    {
        return "order::attributes.orders";
    }
}
