<?php

namespace Modules\Setting\Http\Requests\Api\V1;

use Illuminate\Validation\Rule;
use Modules\Core\Http\Requests\Request;
use Modules\Currency\Currency;
use Modules\Currency\Enums\ExchangeService;
use Modules\Setting\Enums\SettingSection;
use Modules\Support\Country;
use Modules\Support\DateFormats;
use Modules\Support\Enums\Day;
use Modules\Support\Enums\FilesystemDisk;
use Modules\Support\Enums\Frequency;
use Modules\Support\Enums\MailEncryptionProtocol;
use Modules\Support\Enums\Mailer;
use Modules\Support\Locale;
use Modules\Support\TimeFormats;
use Modules\Support\TimeZone;

/**
 * @property SettingSection $section
 */
class SaveSettingRequest extends Request
{
    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        return match ($this->section) {
            SettingSection::General => [
                "supported_countries" => "required|array",
                "supported_countries.*" => ["required", Rule::in(Country::codes())],
                'default_country' => 'required|in_array:supported_countries.*',
                "supported_locales" => "required|array",
                'supported_locales.*' => ['required', Rule::in(Locale::codes())],
                'default_locale' => 'required|in_array:supported_locales.*',
                'default_timezone' => ['required', Rule::in(TimeZone::keys())],
                "default_date_format" => ['required', Rule::in(DateFormats::keys())],
                "default_time_format" => ['required', Rule::in(TimeFormats::keys())],
                "start_of_week" => ["required", Rule::enum(Day::class)],
                "end_of_week" => ["required", Rule::enum(Day::class)],
            ],
            SettingSection::Application => [
                ...$this->getTranslationRules(["app_name" => "required|min:1|max:200"])
            ],
            SettingSection::Currency => [
                "supported_currencies" => "required|array",
                'supported_currencies.*' => ['required', Rule::in(Currency::codes())],
                'default_currency' => 'required|in_array:supported_currencies.*',
                "currency_rate_exchange_service" => ["nullable", Rule::enum(ExchangeService::class)],
                'fixer_access_key' => 'required_if:currency_rate_exchange_service,fixer',
                'forge_api_key' => 'required_if:currency_rate_exchange_service,forge',
                'currency_data_feed_api_key' => 'required_if:currency_rate_exchange_service,currency_data_feed',
                'auto_refresh_currency_rates' => 'required|boolean',
                'auto_refresh_currency_rate_frequency' => ['required_if:auto_refresh_currency_rates,1', 'nullable', Rule::enum(Frequency::class)],
            ],
            SettingSection::Mail => [
                "mail_mailer" => ['required', Rule::enum(Mailer::class)],
                "mail_from_address" => "required|email",
                "mail_from_name" => "required|string|min:1|max:200",
                "mail_host" => "required_if:mail_mailer,smtp|string",
                'mail_port' => "required_if:mail_mailer,smtp|integer|between:1,65535",
                "mail_username" => "nullable|string",
                "mail_password" => "nullable|string",
                "mail_encryption" => ["required_if:mail_mailer,smtp", Rule::enum(MailEncryptionProtocol::class)]
            ],
            SettingSection::Filesystem => [
                "default_filesystem_disk" => ["required", Rule::enum(FilesystemDisk::class)],
                "filesystem_s3_use_path_style_endpoint" => "required_if:default_filesystem_disk,s3|nullable|boolean",
                "filesystem_s3_url" => "required_if:default_filesystem_disk,s3|nullable|string|url",
                "filesystem_s3_endpoint" => "required_if:default_filesystem_disk,s3|nullable|string",
                "filesystem_s3_region" => "required_if:default_filesystem_disk,s3|nullable|string",
                "filesystem_s3_bucket" => "required_if:default_filesystem_disk,s3|nullable|string",
                "encryptable.filesystem_s3_key" => "required_if:default_filesystem_disk,s3|nullable|string",
                "encryptable.filesystem_s3_secret" => "required_if:default_filesystem_disk,s3|nullable|string",
            ],
        };
    }

    /** @inheritDoc */
    protected function availableAttributes(): string
    {
        return "setting::attributes.settings";
    }
}
