<?php

namespace Modules\Printer\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Modules\ActivityLog\Traits\HasActivityLog;
use Modules\Branch\Traits\HasBranch;
use Modules\Pos\Models\PosRegister;
use Modules\Printer\Enum\PrinterConnectionType;
use Modules\Printer\Enum\PrinterRole;
use Modules\Support\Eloquent\Model;
use Modules\Support\Traits\HasActiveStatus;
use Modules\Support\Traits\HasCreatedBy;
use Modules\Support\Traits\HasFilters;
use Modules\Support\Traits\HasSortBy;
use Modules\Support\Traits\HasTagsCache;
use Modules\Translation\Traits\Translatable;


/**
 * @property int $id
 * @property string $name
 * @property PrinterRole $role
 * @property PrinterConnectionType $connection_type
 * @property array|null $options
 * @property array|null $category_slugs
 * @property int|null $pos_register_id
 * @property-read PosRegister|null $posRegister
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 */
class Printer extends Model
{
    use HasActiveStatus,
        HasActivityLog,
        HasCreatedBy,
        HasTagsCache,
        HasSortBy,
        HasFilters,
        Translatable,
        HasBranch;

    /**
     * The attributes that are mass assignable.
     */
    protected $fillable = [
        "name",
        "role",
        "category_slugs",
        "connection_type",
        "options",
        "pos_register_id",
        self::ACTIVE_COLUMN_NAME,
        self::BRANCH_COLUMN_NAME
    ];

    /**
     * The attributes that are translatable.
     *
     * @var array
     */
    protected array $translatable = ['name'];

    /**
     * Perform any actions required after the model boots.
     *
     * @return void
     */
    protected static function booted(): void
    {
        static::saving(function (Printer $printer) {
            if ($printer->role === PrinterRole::Cashier) {
                $printer->category_slugs = null;
            } else {
                $printer->pos_register_id = null;
            }
        });
    }

    /** @inheritDoc */
    public function allowedFilterKeys(): array
    {
        return [
            "search",
            "from",
            "to",
            "role",
            "connection_type",
            self::ACTIVE_COLUMN_NAME,
            self::BRANCH_COLUMN_NAME,
        ];
    }

    /**
     * Scope a query to search across all fields.
     *
     * @param Builder $query
     * @param string $value
     * @return void
     */
    public function scopeSearch(Builder $query, string $value): void
    {
        $query->whereLikeTranslation('name', $value);
    }

    /**
     * Get pos register
     *
     * @return BelongsTo
     */
    public function posRegister(): BelongsTo
    {
        return $this->belongsTo(PosRegister::class)
            ->withOutGlobalBranchPermission()
            ->withoutGlobalActive()
            ->withTrashed();
    }

    /** @inheritDoc */
    protected function getSortableAttributes(): array
    {
        return [
            "name",
            "role",
            "connection_type",
            self::ACTIVE_COLUMN_NAME,
            self::BRANCH_COLUMN_NAME,
        ];
    }

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'role' => PrinterRole::class,
            'connection_type' => PrinterConnectionType::class,
            'category_slugs' => "array",
            'options' => "array",
            self::ACTIVE_COLUMN_NAME => "boolean",
        ];
    }
}
