<?php

namespace Modules\Order\Http\Controllers\Api\V1;

use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Modules\Branch\Models\Branch;
use Modules\Cart\Facades\EditOrderCart;
use Modules\Cart\Facades\PosCart;
use Modules\Core\Http\Controllers\Controller;
use Modules\Order\Enums\OrderPaymentStatus;
use Modules\Order\Enums\OrderStatus;
use Modules\Order\Enums\OrderType;
use Modules\Order\Http\Requests\Api\V1\CancelOrRefundOrderRequest;
use Modules\Order\Http\Requests\Api\V1\CreateOrderRequest;
use Modules\Order\Http\Requests\Api\V1\OrderAddPaymentRequest;
use Modules\Order\Http\Requests\Api\V1\UpdateOrderRequest;
use Modules\Order\Services\CreateOrder\CreateOrderServiceInterface;
use Modules\Order\Services\Order\OrderServiceInterface;
use Modules\Order\Services\UpdateOrder\UpdateOrderServiceInterface;
use Modules\Order\Transformers\Api\V1\ActiveOrderResource;
use Modules\Order\Transformers\Api\V1\EditOrderResource;
use Modules\Order\Transformers\Api\V1\OrderResource;
use Modules\Order\Transformers\Api\V1\ShowOrderResource;
use Modules\Printer\Enum\PrinterRole;
use Modules\Support\ApiResponse;
use Throwable;

class OrderController extends Controller
{
    /**
     * Create a new instance of OrderCreateController
     *
     * @param OrderServiceInterface $service
     * @param CreateOrderServiceInterface $createOrderService
     * @param UpdateOrderServiceInterface $updateOrderService
     */
    public function __construct(
        protected OrderServiceInterface       $service,
        protected CreateOrderServiceInterface $createOrderService,
        protected UpdateOrderServiceInterface $updateOrderService,
    )
    {
    }

    /**
     * This method retrieves and returns a list of Payment models.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function index(Request $request): JsonResponse
    {
        return ApiResponse::pagination(
            paginator: $this->service->get(
                filters: $request->get('filters', []),
                sorts: $request->get('sorts', []),
            ),
            resource: OrderResource::class,
            filters: $request->get('with_filters')
                ? $this->service->getStructureFilters()
                : null
        );
    }

    /**
     * This method cancel order
     *
     * @param CancelOrRefundOrderRequest $request
     * @param int|string $id
     * @return JsonResponse
     */
    public function cancel(CancelOrRefundOrderRequest $request, int|string $id): JsonResponse
    {
        $this->service->cancel($id, $request->validated());

        return ApiResponse::success(
            body: ["success" => true],
            message: __('admin::messages.resources_cancelled', ['resource' => $this->service->label()])
        );
    }

    /**
     * This method Refund order
     *
     * @param CancelOrRefundOrderRequest $request
     * @param int|string $id
     * @return JsonResponse
     */
    public function refund(CancelOrRefundOrderRequest $request, int|string $id): JsonResponse
    {
        $this->service->refund($id, $request->validated());

        return ApiResponse::success(body: ["success" => true], message: __('order::messages.order_refunded'));
    }

    /**
     * This method retrieves and returns a single Payment model based on the provided identifier.
     *
     * @param int|string $id
     * @return JsonResponse
     */
    public function show(int|string $id): JsonResponse
    {
        return ApiResponse::success(
            body: new ShowOrderResource($this->service->show($id))
        );
    }

    /**
     * Print order
     *
     * @param Request $request
     * @param int|string $id
     * @return JsonResponse
     * @throws Exception
     */
    public function print(Request $request, int|string $id): JsonResponse
    {
        $roles = array_filter(array_map(fn($role) => PrinterRole::tryFrom($role), $request->get('roles', [])));

        $this->service->print($id, empty($roles) ? null : $roles);

        return ApiResponse::success(body: ["success" => true], message: __('order::messages.print_success'));
    }

    /**
     * This method stores the provided data into storage for the Order model.
     *
     * @param CreateOrderRequest $request
     * @return JsonResponse
     * @throws Throwable
     */
    public function store(CreateOrderRequest $request): JsonResponse
    {
        $order = $this->createOrderService->create($request->all());

        return ApiResponse::created(
            body: [
                "order_id" => $order->reference_no,
                "cart" => PosCart::instance()
            ],
            resource: $this->service->label()
        );
    }

    /**
     * This method update the provided data into storage for the Order model.
     *
     * @param UpdateOrderRequest $request
     * @param string|int $id
     * @return JsonResponse
     * @throws Throwable
     */
    public function update(UpdateOrderRequest $request, string|int $id): JsonResponse
    {
        $order = $this->updateOrderService->update($id, $request->all());

        return ApiResponse::updated(
            body: [
                "order_id" => $order->reference_no,
                "cart" => EditOrderCart::instance()
            ],
            resource: $this->service->label()
        );
    }

    /**
     * Get add payment meta
     *
     * @param int|string $id
     * @return JsonResponse
     */
    public function getAddPaymentMeta(int|string $id): JsonResponse
    {
        return ApiResponse::success($this->service->getAddPaymentMeta($id));
    }

    /**
     * Get update status meta
     *
     * @param int|string $id
     * @return JsonResponse
     */
    public function getUpdateStatusMeta(int|string $id): JsonResponse
    {
        return ApiResponse::success($this->service->getUpdateStatusMeta($id));
    }

    /**
     * Order add payment
     *
     * @param OrderAddPaymentRequest $request
     * @param int|string $orderId
     * @return JsonResponse
     * @throws Throwable
     */
    public function addPayment(OrderAddPaymentRequest $request, int|string $orderId): JsonResponse
    {
        $this->service->addPayment($orderId, $request->validated());

        return ApiResponse::success(message: __("order::messages.order_paid_successfully"));
    }

    /**
     * Order add payment merge
     *
     * @param OrderAddPaymentRequest $request
     * @param int $mergeId
     * @return JsonResponse
     * @throws Throwable
     */
    public function addPaymentMerge(OrderAddPaymentRequest $request, int $mergeId): JsonResponse
    {
        $this->service->addPaymentMerge($mergeId, $request->validated());

        return ApiResponse::success(message: __("order::messages.order_paid_successfully"));
    }


    /**
     * Get active orders
     *
     * @return JsonResponse
     */
    public function activeOrders(): JsonResponse
    {
        $user = auth()->user();

        $branch = $user->assignedToBranch() ? $user->branch : Branch::main()->first();

        return ApiResponse::success(
            [
                "orders" => ActiveOrderResource::collection($this->service->activeOrders()),
                "filters" => [
                    "statuses" => OrderStatus::toArrayTrans([
                        OrderStatus::Refunded->value,
                        OrderStatus::Cancelled->value,
                        OrderStatus::Merged->value,
                        OrderStatus::Served->value,
                    ]),
                    "order_types" => array_filter(
                        OrderType::toArrayTrans(),
                        fn($orderType) => $orderType['id'] != OrderType::DineIn->value && in_array($orderType['id'], $branch->order_types)
                    ),
                    "payment_statuses" => OrderPaymentStatus::toArrayTrans()
                ]
            ]
        );
    }

    /**
     * Get upcoming orders
     *
     * @return JsonResponse
     */
    public function upcomingOrders(): JsonResponse
    {
        $user = auth()->user();

        $branch = $user->effective_branch;

        return ApiResponse::success(
            [
                "orders" => ActiveOrderResource::collection($this->service->upcomingOrders()),
                "filters" => [
                    "statuses" => OrderStatus::toArrayTrans([
                        OrderStatus::Refunded->value,
                        OrderStatus::Cancelled->value,
                        OrderStatus::Merged->value,
                        OrderStatus::Completed->value,
                        OrderStatus::Served->value,
                        OrderStatus::Ready->value,
                        OrderStatus::Preparing->value,
                    ]),
                    "order_types" => array_filter(
                        OrderType::toArrayTrans(),
                        fn($orderType) => in_array($orderType['id'], [OrderType::Catering->value, OrderType::PreOrder->value]) && in_array($orderType['id'], $branch->order_types)
                    ),
                    "payment_statuses" => OrderPaymentStatus::toArrayTrans()
                ]
            ]
        );
    }

    /**
     * Move to next status
     *
     * @param int|string $orderId
     * @return JsonResponse
     * @throws Throwable
     */
    public function moveToNextStatus(int|string $orderId): JsonResponse
    {
        $newStatus = $this->service->moveToNextStatus($orderId);

        return ApiResponse::success(message: __("order::messages.order_update_status_to_successfully", ['status' => $newStatus->trans()]));
    }

    /**
     * Get for edit meta
     *
     * @param int|string $orderId
     * @return JsonResponse
     * @throws Throwable
     */
    public function getFormEditMeta(int|string $orderId): JsonResponse
    {
        $meta = $this->service->getFormEditMeta($orderId);

        return ApiResponse::success([...$meta, "order" => new EditOrderResource($meta['order'])]);
    }

}
